<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2025 Teclib' and contributors.
 * @copyright 2003-2014 by the INDEPNET Development Team.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

namespace Glpi\Form\Destination;

use CommonDBTM;
use Glpi\DBAL\JsonFieldInterface;
use Glpi\Form\AnswersSet;
use Glpi\Form\Destination\CommonITILField\Category;
use Glpi\Form\Form;

interface DestinationFieldInterface
{
    /**
     * Get the unique key used to set/get this field configuration in the
     * destination JSON configuration.
     *
     * @return string
     */
    public static function getKey(): string;

    /**
     * Label to be displayed when configuring this field.
     *
     * @return string
     */
    public function getLabel(): string;

    /**
     * Render the input field for this configuration.
     *
     * @param Form       $form
     * @param JsonFieldInterface $config
     * @param string     $input_name Input name supplied by the controller.
     *                               Must be reused in the actual input field.
     * @param array      $display_options Common twig options to display the
     *                                    input.
     *
     * @return string
     */
    public function renderConfigForm(
        Form $form,
        FormDestination $destination,
        JsonFieldInterface $config,
        string $input_name,
        array $display_options
    ): string;

    /**
     * Apply configurated value to the given input.
     *
     * @param JsonFieldInterface $config
     * @param array               $input
     * @param AnswersSet          $answers_set
     *
     * @return array
     */
    public function applyConfiguratedValueToInputUsingAnswers(
        JsonFieldInterface $config,
        array $input,
        AnswersSet $answers_set
    ): array;

    /**
     * Process field after all destinations have been created
     *
     * @param JsonFieldInterface $config The field configuration
     * @param AnswersSet $answers_set The answers from the form
     * @param array<int, CommonDBTM[]> $created_objects Array mapping destination_id to created objects
     * @return void
     */
    public function applyConfiguratedValueAfterDestinationCreation(
        FormDestination $destination,
        JsonFieldInterface $config,
        AnswersSet $answers_set,
        array $created_objects
    ): void;

    /**
     * Check if this field support auto generated value.
     * If true, getAutoGeneratedValue() will be called to get the value.
     *
     * @return bool
     */
    public function supportAutoConfiguration(): bool;

    /**
     * Get the auto generated config for this field.
     *
     * @param Form $form
     *
     * @return JsonFieldInterface|null (null if not supported)
     */
    public function getAutoGeneratedConfig(Form $form, ?AnswersSet $answers_set): ?JsonFieldInterface;

    /**
     * Get the weight of this config field.
     * The weight is used to sort fields in the form destination configuration.
     *
     * @return int
     */
    public function getWeight(): int;

    public function getDefaultConfig(Form $form): mixed;

    /** @return class-string<JsonFieldInterface>  */
    public function getConfigClass(): string;

    /**
     * Prepare input data before saving it to the database.
     *
     * @param array $input
     * @return array
     */
    public function prepareInput(array $input): array;

    /**
     * Get the possible values for the main configuration dropdown.
     *
     * @return array
     */
    public function getStrategiesForDropdown(): array;

    /**
     * Check if this field can have multiple strategies at the same time.
     *
     * @return bool
     */
    public function canHaveMultipleStrategies(): bool;

    public function getCategory(): Category;
}
